// g_combat.c

#include "g_local.h"
#include "m_player.h"


/*
============
CanDamage

Returns true if the inflictor can directly damage the target.  Used for
explosions and melee attacks.
============
*/
qboolean CanDamage (edict_t *targ, edict_t *inflictor)
{
	vec3_t	dest;
	trace_t	trace;

// bmodels need special checking because their origin is 0,0,0
	if (targ->movetype == MOVETYPE_PUSH)
	{
		VectorAdd (targ->absmin, targ->absmax, dest);
		VectorScale (dest, 0.5, dest);
		trace = gi.trace (inflictor->s.origin, vec3_origin, vec3_origin, dest, inflictor, MASK_SOLID);
		if (trace.fraction == 1.0)
			return true;
		if (trace.ent == targ)
			return true;
		return false;
	}
	
	trace = gi.trace (inflictor->s.origin, vec3_origin, vec3_origin, targ->s.origin, inflictor, MASK_SOLID);
	if (trace.fraction == 1.0)
		return true;

	VectorCopy (targ->s.origin, dest);
	dest[0] += 15.0;
	dest[1] += 15.0;
	trace = gi.trace (inflictor->s.origin, vec3_origin, vec3_origin, dest, inflictor, MASK_SOLID);
	if (trace.fraction == 1.0)
		return true;

	VectorCopy (targ->s.origin, dest);
	dest[0] += 15.0;
	dest[1] -= 15.0;
	trace = gi.trace (inflictor->s.origin, vec3_origin, vec3_origin, dest, inflictor, MASK_SOLID);
	if (trace.fraction == 1.0)
		return true;

	VectorCopy (targ->s.origin, dest);
	dest[0] -= 15.0;
	dest[1] += 15.0;
	trace = gi.trace (inflictor->s.origin, vec3_origin, vec3_origin, dest, inflictor, MASK_SOLID);
	if (trace.fraction == 1.0)
		return true;

	VectorCopy (targ->s.origin, dest);
	dest[0] -= 15.0;
	dest[1] -= 15.0;
	trace = gi.trace (inflictor->s.origin, vec3_origin, vec3_origin, dest, inflictor, MASK_SOLID);
	if (trace.fraction == 1.0)
		return true;


	return false;
}


/*
============
Killed
============
*/
void Killed (edict_t *targ, edict_t *inflictor, edict_t *attacker, int damage, vec3_t point)
{
	if (targ->health < -999)
		targ->health = -999;

	if (attacker)
		targ->enemy = attacker;

	if ((targ->svflags & SVF_MONSTER) && (targ->deadflag != DEAD_DEAD))
	{
		targ->svflags |= SVF_DEADMONSTER;	// now treat as a different content type
		if (!(targ->monsterinfo.aiflags & AI_GOOD_GUY) && attacker->client)
		{
			level.killed_monsters++;
			if (coop->value && attacker->client)
			{
				attacker->client->resp.score++;
				if (headShot)
					attacker->client->resp.headshots++;
			}
			if (!deathmatch->value && !coop->value)
			{
				attacker->client->resp.bullettime+=8;
			}
			// medics won't heal monsters that they kill themselves
			if (strcmp(attacker->classname, "monster_medic") == 0)
				targ->owner = attacker;
		}
	}

	if (targ->movetype == MOVETYPE_PUSH || targ->movetype == MOVETYPE_STOP || targ->movetype == MOVETYPE_NONE)
	{	// doors, triggers, etc
		targ->die (targ, inflictor, attacker, damage, point);
		return;
	}

	if ((targ->svflags & SVF_MONSTER) && (targ->deadflag != DEAD_DEAD))
	{
		targ->touch = NULL;
		monster_death_use (targ);
	}

	targ->die (targ, inflictor, attacker, damage, point);
}


/*
================
SpawnDamage
================
*/
void SpawnDamage (int type, vec3_t origin, vec3_t normal, int damage)
{
	if (damage > 255)
		damage = 255;
	gi.WriteByte (svc_temp_entity);
	gi.WriteByte (type);
//	gi.WriteByte (damage);
	gi.WritePosition (origin);
	gi.WriteDir (normal);
	gi.multicast (origin, MULTICAST_PVS);
}


/*
============
T_Damage

targ		entity that is being damaged
inflictor	entity that is causing the damage
attacker	entity that caused the inflictor to damage targ
	example: targ=monster, inflictor=rocket, attacker=player

dir			direction of the attack
point		point at which the damage is being inflicted
normal		normal vector from that point
damage		amount of damage being inflicted
knockback	force to be applied against targ as a result of the damage

dflags		these flags are used to control how T_Damage works
	DAMAGE_RADIUS			damage was indirect (from a nearby explosion)
	DAMAGE_NO_ARMOR			armor does not protect from this damage
	DAMAGE_ENERGY			damage is from an energy based weapon
	DAMAGE_NO_KNOCKBACK		do not affect velocity, just view angles
	DAMAGE_BULLET			damage is from a bullet (used for ricochets)
	DAMAGE_NO_PROTECTION	kills godmode, armor, everything
============
*/
static int CheckPowerArmor (edict_t *ent, vec3_t point, vec3_t normal, int damage, int dflags)
{
	gclient_t	*client;
	int			save;
	int			power_armor_type;
	int			index;
	int			damagePerCell;
	int			pa_te_type;
	int			power;
	int			power_used;

	if (!damage)
		return 0;

	client = ent->client;

	if (dflags & DAMAGE_NO_ARMOR)
		return 0;

	if (client)
	{
		power_armor_type = PowerArmorType (ent);
		if (power_armor_type != POWER_ARMOR_NONE)
		{
			index = ITEM_INDEX(FindItem("Cells"));
			power = client->pers.inventory[index];
		}
	}
	else if (ent->svflags & SVF_MONSTER)
	{
		power_armor_type = ent->monsterinfo.power_armor_type;
		power = ent->monsterinfo.power_armor_power;
	}
	else
		return 0;

	if (power_armor_type == POWER_ARMOR_NONE)
		return 0;
	if (!power)
		return 0;

	if (power_armor_type == POWER_ARMOR_SCREEN)
	{
		vec3_t		vec;
		float		dot;
		vec3_t		forward;

		// only works if damage point is in front
		AngleVectors (ent->s.angles, forward, NULL, NULL);
		VectorSubtract (point, ent->s.origin, vec);
		VectorNormalize (vec);
		dot = DotProduct (vec, forward);
		if (dot <= 0.3)
			return 0;

		damagePerCell = 1;
		pa_te_type = TE_SCREEN_SPARKS;
		damage = damage / 3;
	}
	else
	{
		damagePerCell = 2;
		pa_te_type = TE_SHIELD_SPARKS;
		damage = (2 * damage) / 3;
	}

	save = power * damagePerCell;
	if (!save)
		return 0;
	if (save > damage)
		save = damage;

	SpawnDamage (pa_te_type, point, normal, save);
	ent->powerarmor_time = level.time + 0.2;

	power_used = save / damagePerCell;

	if (client)
		client->pers.inventory[index] -= power_used;
	else
		ent->monsterinfo.power_armor_power -= power_used;
	return save;
}

static int CheckArmor (edict_t *ent, vec3_t point, vec3_t normal, int damage, int te_sparks, int dflags)
{
	gclient_t	*client;
	int			save;
	int			index;
	gitem_t		*armor;

	if (!damage)
		return 0;

	client = ent->client;

	if (!client)
		return 0;

	if (dflags & DAMAGE_NO_ARMOR)
		return 0;

	index = ArmorIndex (ent);
	if (!index)
		return 0;

	armor = GetItemByIndex (index);

	if (dflags & DAMAGE_ENERGY)
		save = ceil(((gitem_armor_t *)armor->info)->energy_protection*damage);
	else
		save = ceil(((gitem_armor_t *)armor->info)->normal_protection*damage);
	if (save >= client->pers.inventory[index])
		save = client->pers.inventory[index];

	if (!save)
		return 0;

	client->pers.inventory[index] -= save;
	SpawnDamage (te_sparks, point, normal, save);

	return save;
}

void M_ReactToDamage (edict_t *targ, edict_t *attacker)
{
	if (!(attacker->client) && !(attacker->svflags & SVF_MONSTER))
		return;

	if (attacker == targ || attacker == targ->enemy)
		return;

	// if we are a good guy monster and our attacker is a player
	// or another good guy, do not get mad at them
	if (targ->monsterinfo.aiflags & AI_GOOD_GUY)
	{
		if (attacker->client || (attacker->monsterinfo.aiflags & AI_GOOD_GUY))
			return;
	}

	// we now know that we are not both good guys

	// if attacker is a client, get mad at them because he's good and we're not
	if (attacker->client)
	{
		targ->monsterinfo.aiflags &= ~AI_SOUND_TARGET;

		// this can only happen in coop (both new and old enemies are clients)
		// only switch if can't see the current enemy
		if (targ->enemy && targ->enemy->client)
		{
			if (visible(targ, targ->enemy))
			{
				targ->oldenemy = attacker;
				return;
			}
			targ->oldenemy = targ->enemy;
		}
		targ->enemy = attacker;
		if (!(targ->monsterinfo.aiflags & AI_DUCKED))
			FoundTarget (targ);
		return;
	}

	// it's the same base (walk/swim/fly) type and a different classname and it's not a tank
	// (they spray too much), get mad at them
	if (((targ->flags & (FL_FLY|FL_SWIM)) == (attacker->flags & (FL_FLY|FL_SWIM))) &&
		 (strcmp (targ->classname, attacker->classname) != 0) &&
		 (strcmp(attacker->classname, "monster_tank") != 0) &&
		 (strcmp(attacker->classname, "monster_supertank") != 0) &&
		 (strcmp(attacker->classname, "monster_makron") != 0) &&
		 (strcmp(attacker->classname, "monster_jorg") != 0) )
	{
		if (targ->enemy && targ->enemy->client)
			targ->oldenemy = targ->enemy;
		targ->enemy = attacker;
		if (!(targ->monsterinfo.aiflags & AI_DUCKED))
			FoundTarget (targ);
	}
	// if they *meant* to shoot us, then shoot back
	else if (attacker->enemy == targ)
	{
		if (targ->enemy && targ->enemy->client)
			targ->oldenemy = targ->enemy;
		targ->enemy = attacker;
		if (!(targ->monsterinfo.aiflags & AI_DUCKED))
			FoundTarget (targ);
	}
	// otherwise get mad at whoever they are mad at (help our buddy) unless it is us!
	else if (attacker->enemy && attacker->enemy != targ)
	{
		if (targ->enemy && targ->enemy->client)
			targ->oldenemy = targ->enemy;
		targ->enemy = attacker->enemy;
		if (!(targ->monsterinfo.aiflags & AI_DUCKED))
			FoundTarget (targ);
	}
}

qboolean CheckTeamDamage (edict_t *targ, edict_t *attacker)
{
		//FIXME make the next line real and uncomment this block
		// if ((ability to damage a teammate == OFF) && (targ's team == attacker's team))
	return false;
}

qboolean IsHeadMonster (edict_t *ent)
{
	if (ent->svflags & SVF_MONSTER)
		if 	(
		   !Q_stricmp (ent->classname, "monster_berserk")
//		|| !Q_stricmp (ent->classname, "monster_gladiator") //gladiators too strong
		|| !Q_stricmp (ent->classname, "monster_gunner")
		|| !Q_stricmp (ent->classname, "monster_infantry")
		|| !Q_stricmp (ent->classname, "monster_soldier_light")
		|| !Q_stricmp (ent->classname, "monster_soldier")
		|| !Q_stricmp (ent->classname, "monster_soldier_ss")
		|| !Q_stricmp (ent->classname, "turret_driver")
		|| !Q_stricmp (ent->classname, "monster_medic")
		|| !Q_stricmp (ent->classname, "monster_chick")
		|| !Q_stricmp (ent->classname, "monster_brain")
		|| !Q_stricmp (ent->classname, "monster_mutant")
		|| !Q_stricmp (ent->classname, "monster_flyer")
		|| !Q_stricmp (ent->classname, "misc_insane")
		|| !Q_stricmp (ent->classname, "monster_turret_driver")
		)
			return true;
	return false;
}
void T_Damage (edict_t *targ, edict_t *inflictor, edict_t *attacker, vec3_t dir, vec3_t point, vec3_t normal, int damage, int knockback, int dflags, int mod)
{
	gclient_t	*client;
	int			take;
	int			save;
	int			asave;
	int			psave;
	int			te_sparks;

	if (!targ->takedamage)
		return;

	// friendly fire avoidance
	// if enabled you can't hurt teammates (but you can hurt yourself)
	// knockback still occurs
	if ((targ != attacker) && ((deathmatch->value && ((int)(dmflags->value) & (DF_MODELTEAMS | DF_SKINTEAMS))) || coop->value || sv_teams->value))
	{
		if (OnSameTeam (targ, attacker))
		{
			if ((int)(dmflags->value) & DF_NO_FRIENDLY_FIRE)
				damage = 0;
			else
				mod |= MOD_FRIENDLY_FIRE;
		}
	}

	meansOfDeath = mod;

	// easy mode takes half damage
	if (skill->value == 0 && deathmatch->value == 0 && targ->client)
	{
		damage *= 0.5;
		if (!damage)
			damage = 1;
	}

	if (coop->value && sv_teams->value)
		if (targ->client&&attacker->client)
			damage = 0;

	client = targ->client;

	if (dflags & DAMAGE_BULLET)
		te_sparks = TE_BULLET_SPARKS;
	else
		te_sparks = TE_SPARKS;

	VectorNormalize(dir);

// bonus damage for suprising a monster
	if (!(dflags & DAMAGE_RADIUS) && (targ->svflags & SVF_MONSTER) && (attacker->client) && (!targ->enemy) && (targ->health > 0))
		damage *= 1 + (4 - skill->value)/2;	//damage *= 2 //change for more in harder levels

	if (targ->flags & FL_NO_KNOCKBACK)
		knockback = 0;

// figure momentum add
	if (!(dflags & DAMAGE_NO_KNOCKBACK))
	{
		if ((knockback) && (targ->movetype != MOVETYPE_NONE) && (targ->movetype != MOVETYPE_BOUNCE) && (targ->movetype != MOVETYPE_PUSH) && (targ->movetype != MOVETYPE_STOP))
		{
			vec3_t	kvel;
			float	mass, kick_limit = 750;

			if (targ->mass < 50)
				mass = 50;
			else
				mass = targ->mass;

			if (targ->client  && attacker == targ)
				VectorScale (dir, 1600.0 * (float)knockback / mass, kvel);	// the rocket jump hack...
			else
				VectorScale (dir, 500.0 * (float)knockback / mass, kvel);

			//limit code begin
			if (VectorLength(kvel)>kick_limit) //kick_limit is limit #
			{
				VectorNormalize(kvel);
				VectorScale(kvel, kick_limit, kvel);
			}
			//limit code end

			VectorAdd (targ->velocity, kvel, targ->velocity);
		}
	}

	take = damage;
	save = 0;

	// check for godmode
	if ( (targ->flags & FL_GODMODE) && !(dflags & DAMAGE_NO_PROTECTION) )
	{
		take = 0;
		save = damage;
		SpawnDamage (te_sparks, point, normal, save);
	}

	// check for invincibility
	if ((client && client->invincible_framenum > level.framenum ) && !(dflags & DAMAGE_NO_PROTECTION))
	{
		if (targ->pain_debounce_time < level.time)
		{
			gi.sound(targ, CHAN_ITEM, gi.soundindex("items/protect4.wav"), 1, ATTN_NORM, 0);
			targ->pain_debounce_time = level.time + 2;
		}
		take = 0;
		save = damage;
	}

	psave = CheckPowerArmor (targ, point, normal, take, dflags);
	take -= psave;

	asave = CheckArmor (targ, point, normal, take, te_sparks, dflags);
	take -= asave;

	//treat cheat/powerup savings the same as armor
	asave += save;

	// team damage avoidance
	if (!(dflags & DAMAGE_NO_PROTECTION) && CheckTeamDamage (targ, attacker))
		return;

// do the damage
	if (take)
	{
		qboolean head_damage = false, leg_damage = false;

		if ((targ->svflags & SVF_MONSTER) || (client))
			SpawnDamage (TE_BLOOD, point, normal, take);
		else
			SpawnDamage (te_sparks, point, normal, take);

		//**************************************************
		//***********   locational damage code   ***********
		//**************************************************	
		headShot=0;
		if (!Q_stricmp (targ->classname, "monster_supertank")||	!Q_stricmp (targ->classname, "monster_boss2") ||
			!Q_stricmp (targ->classname, "monster_boss3_stand") ||	!Q_stricmp (targ->classname, "monster_makron") ||
			!Q_stricmp (targ->classname, "monster_jorg") ) //super tough!!!
		{
			if (dflags&DAMAGE_RADIUS)
				take*=0.5;
			if (dflags&DAMAGE_ENERGY)
				take*=0.25;
			if (!(dflags&DAMAGE_NO_ARMOR))
				take*=0.33333333333333333;
		}		
		else if (targ->health>0)
			if ( ((targ->client)||IsHeadMonster(targ)) && (dflags!=DAMAGE_RADIUS) )
			{
				vec3_t		vec;
				float		dot;
				vec3_t		forward;
				int HeadOffset = 0;
				int monsterheight = abs(targ->mins[2]-targ->maxs[2]);
					//Check to see if it is a tall enough monster to hurt that way
				AngleVectors (targ->s.angles, NULL, NULL, forward);
				VectorSubtract (point, targ->s.origin, vec);
				VectorNormalize (vec);
				dot = DotProduct (vec, forward);

				if (targ->s.frame>134 && targ->health>0 && !IsHeadMonster(targ)) //is crouching
				{
					if (dot >= 0.50)
						head_damage = true;
					else if (dot <= 0.15)
						leg_damage = true;
					HeadOffset = targ->viewheight;
				}
				else if (targ->health>0 && !IsHeadMonster(targ))
				{
					if (dot >= 0.85)
						head_damage = true;
					else if (dot <= 0.35)
						leg_damage = true;
					HeadOffset = targ->viewheight;
				}
				else if (targ->health>0 && IsHeadMonster(targ))
				{
					if (dot >= 0.85)
						head_damage = true;
					else if (dot <= 0.35)
						leg_damage = true;
					HeadOffset = 30;
				}

				if (leg_damage&&!targ->waterlevel)
				{
					take /= 2;	//Reduce damage taken here now
				}
				else if (head_damage)
				{
					vec3_t dir, start;
					start[0]=targ->s.origin[0];	start[1]=targ->s.origin[1];	start[2]=targ->s.origin[2]+HeadOffset;
					VectorClear(dir);
					//Multiply damage taken here now
					take *= 2;	
						gi.sound(targ, CHAN_VOICE, gi.soundindex("chick/chkatck4.wav"), 1, ATTN_NORM, 0);
						gi.sound(targ, CHAN_WEAPON, gi.soundindex("world/flesh1.wav"), 1, ATTN_NORM, 0);
						gi.sound(targ, CHAN_ITEM, gi.soundindex("world/flesh2.wav"), 1, ATTN_NORM, 0);
						gi.sound(targ, CHAN_BODY, gi.soundindex("makron/brain1.wav"), 1, ATTN_NORM, 0);
					headShot=1;
					//Make pretty blood splat :)
					dir[0]=1;	dir[1]=0;	dir[2]=2;
					gi.WriteByte (svc_temp_entity);
					gi.WriteByte (TE_BLOOD);
					gi.WritePosition (start);
					gi.WriteDir (dir);
					gi.multicast (start, MULTICAST_PVS);
					//Make more blood splat :)
					dir[0]=0;	dir[1]=1;	dir[2]=2;
					gi.WriteByte (svc_temp_entity);
					gi.WriteByte (TE_BLOOD);
					gi.WritePosition (start);
					gi.WriteDir (dir);
					gi.multicast (start, MULTICAST_PVS);
					//Make much more blood splat :)
					dir[0]=-1;	dir[1]=-1;	dir[2]=2;
					gi.WriteByte (svc_temp_entity);
					gi.WriteByte (TE_BLOOD);
					gi.WritePosition (start);
					gi.WriteDir (dir);
					gi.multicast (start, MULTICAST_PVS);
				}

			}	//*/
		//**************************************************
		//**************   head damage code   **************
		//**************************************************
		targ->health = targ->health - take;
			
		if (targ->client && sv_damageslow->value)
		{
			float slowadd = take/7.5 ;

			if (leg_damage)
				targ->client->damage_div *= 3;			
			else if (slowadd>2)
				slowadd=2;

			targ->client->damage_div += slowadd;
			if (targ->client->damage_div>3)
				targ->client->damage_div=3;
		}

		if (targ->health <= 0)
		{
			if ((targ->svflags & SVF_MONSTER) || (client))
				targ->flags |= FL_NO_KNOCKBACK;
			Killed (targ, inflictor, attacker, take, point);
			return;
		}
	}

	if (targ->svflags & SVF_MONSTER)
	{
		M_ReactToDamage (targ, attacker);
		if (!(targ->monsterinfo.aiflags & AI_DUCKED) && (take))
		{
			targ->pain (targ, attacker, knockback, take);
			// nightmare mode monsters don't go into pain frames often
			if (skill->value == 3)
				targ->pain_debounce_time = level.time + 5;
		}
	}
	else if (client)
	{
		
		if (!(targ->flags & FL_GODMODE) && (take))
			targ->pain (targ, attacker, knockback, take);
	}
	else if (take)
	{
		if (targ->pain)
			targ->pain (targ, attacker, knockback, take);
	}

	// add to the damage inflicted on a player this frame
	// the total will be turned into screen blends and view angle kicks
	// at the end of the frame
	if (client)
	{
		client->damage_parmor += psave;
		client->damage_armor += asave;
		client->damage_blood += take;
		client->damage_knockback += knockback;
		VectorCopy (point, client->damage_from);
	}
}


/*
============
T_FlashRadius
============
*/
int distance (edict_t *self, edict_t *other)
{
	vec3_t	v;
	float	len;

	VectorSubtract (self->s.origin, other->s.origin, v);
	len = VectorLength (v);
	if (len < 100)
		return 160;
	if (len < 200)
		return 80;
	if (len < 400)
		return 40;
	return 20;
}

void T_FlashRadius (edict_t *inflictor, edict_t *attacker, float damage, edict_t *ignore, float radius)
{
	float	points;
	edict_t	*ent = NULL;
	vec3_t	v;
	vec3_t	dir;
	int limit = 70;

	//while ((ent = findradius(ent, inflictor->s.origin, radius)) != NULL)
	while ((ent = findradius(ent, inflictor->s.origin, radius)) != NULL)
	{
		if (ent == ignore)
			continue;
		if (!ent->takedamage)
			continue;

		VectorAdd (ent->mins, ent->maxs, v);
		VectorMA (ent->s.origin, 0.5, v, v);
		VectorSubtract (inflictor->s.origin, v, v);
		points = damage - 0.5 * VectorLength (v);
		if (ent == attacker)
			points = points * 0.5;
		if (points > 0)
		{
			if (CanDamage (ent, inflictor))
			{
				VectorSubtract (ent->s.origin, inflictor->s.origin, dir);

				if (!infront (ent ,inflictor))
					ent->flashbanged+=30;
				else
					ent->flashbanged+=30+distance(ent, inflictor);

				ent->flashbanged = (ent->flashbanged>limit)?limit:ent->flashbanged;
			}
		}
	}
}


/*
============
T_RadiusDamage
============
*/
void T_RadiusDamage (edict_t *inflictor, edict_t *attacker, float damage, edict_t *ignore, float radius, int mod)
{
	float	points;
	edict_t	*ent = NULL;
	vec3_t	v;
	vec3_t	dir;

	while ((ent = findradius(ent, inflictor->s.origin, radius)) != NULL)
	{
		if (ent == ignore)
			continue;
		if (!ent->takedamage)
			continue;

		VectorAdd (ent->mins, ent->maxs, v);
		VectorMA (ent->s.origin, 0.5, v, v);
		VectorSubtract (inflictor->s.origin, v, v);
		points = damage - 0.5 * VectorLength (v);
		if (ent == attacker)
			points = points * 0.5;
		if (points > 0)
		{
			if (CanDamage (ent, inflictor))
			{
				VectorSubtract (ent->s.origin, inflictor->s.origin, dir);
				T_Damage (ent, inflictor, attacker, dir, inflictor->s.origin, vec3_origin, (int)points, (int)points, DAMAGE_RADIUS, mod);
			}
		}
	}
}

void T_RadiusDamageFire (edict_t *inflictor, edict_t *attacker, float damage, edict_t *ignore, float radius, int mod)
{
	float	points;
	edict_t	*ent = NULL;
	vec3_t	v;
	vec3_t	dir;

	while ((ent = findradius(ent, inflictor->s.origin, radius)) != NULL)
	{
		if (ent == ignore)
			continue;
		if (!ent->takedamage)
			continue;
		
		if (ent->client)
			if (ent->client->pers.inventory[ITEM_INDEX(FindItem("Body Armor"))])
				continue;
	
		VectorAdd (ent->mins, ent->maxs, v);
		VectorMA (ent->s.origin, 0.5, v, v);
		VectorSubtract (inflictor->s.origin, v, v);
		points = damage;
		if (points > 0)
		{
			if (CanDamage (ent, inflictor))
			{
				VectorSubtract (ent->s.origin, inflictor->s.origin, dir);
				if (ent->client)
				{
					if (!ent->client->goggles)
						T_Damage (ent, inflictor, attacker, dir, inflictor->s.origin, vec3_origin, (int)points, (int)points, DAMAGE_RADIUS, mod);
				}
				else
					T_Damage (ent, inflictor, attacker, dir, inflictor->s.origin, vec3_origin, (int)points, (int)points, DAMAGE_RADIUS, mod);
			}
		}
	}
}

void T_RadiusDamageGas (edict_t *inflictor, edict_t *attacker, float damage, edict_t *ignore, float radius, int mod)
{
	float	points;
	edict_t	*ent = NULL;
	vec3_t	v;
	vec3_t	dir;

	while ((ent = findradius(ent, inflictor->s.origin, radius)) != NULL)
	{
		if (ent->classname)
			if (ent->classname=="flame")
			{
				int damage = 300;
				int radius = 300;
				T_RadiusDamage(inflictor, ent->owner, damage, NULL, radius, MOD_GAS_BOOM);
				bigExplosion(ent->s.origin, vec3_origin, 3);
			}
		
		if (ent == ignore)
			continue;
		if (!ent->takedamage)
			continue;

		VectorAdd (ent->mins, ent->maxs, v);
		VectorMA (ent->s.origin, 0.5, v, v);
		VectorSubtract (inflictor->s.origin, v, v);
		points = damage;
		if (points > 0)
		{
			if (CanDamage (ent, inflictor))
			{
				VectorSubtract (ent->s.origin, inflictor->s.origin, dir);
				if (ent->client)
				{
					if (!ent->client->goggles)
						T_Damage (ent, inflictor, attacker, dir, inflictor->s.origin, vec3_origin, (int)points, (int)points, DAMAGE_RADIUS, mod);
				}
				else
					T_Damage (ent, inflictor, attacker, dir, inflictor->s.origin, vec3_origin, (int)points, (int)points, DAMAGE_RADIUS, mod);
			}
		}
	}
}

void T_RadiusDamageDischarge (edict_t *inflictor, edict_t *attacker, float damage, edict_t *ignore, float radius, int mod)
{
	float	points;
	edict_t	*ent = NULL;
	vec3_t	v;
	vec3_t	dir;

	while ((ent = findradius(ent, inflictor->s.origin, radius)) != NULL)
	{
		if (ent->classname)
			if (ent->classname=="flame")
			{
				int damage = 300;
				int radius = 300;
				T_RadiusDamage(inflictor, ent->owner, damage, NULL, radius, MOD_GAS_BOOM);
				bigExplosion(ent->s.origin, vec3_origin, 3);
			}
		
		if (!ent->takedamage)
			continue;

		VectorAdd (ent->mins, ent->maxs, v);
		VectorMA (ent->s.origin, 0.5, v, v);
		VectorSubtract (inflictor->s.origin, v, v);
		points = damage;
		if (points > 0)
		{

			if ( ((ent->client)||(ent->svflags & SVF_MONSTER)))
				if (CanDamage (ent, inflictor))
				{
					VectorSubtract (ent->s.origin, inflictor->s.origin, dir);
					if (ent->waterlevel)
						T_Damage (ent, inflictor, attacker, dir, inflictor->s.origin, vec3_origin, 
							(ent == ignore)?(int)(points*0.5):(int)points, (ent == ignore)?(int)(points*0.5):(int)points,
							DAMAGE_RADIUS, mod);
				}
		}
	}
}


#define FIRECATCHRADIUS 50

void T_RadiusDamageFireLink (edict_t *inflictor)
{
	float	points;
	float	radius = FIRECATCHRADIUS;
	edict_t	*ent = NULL;
	vec3_t	v;
	vec3_t	dir;

	if (Q_stricmp (inflictor->classname, "tossedflame"))
		return;

	while ((ent = findradius(ent, inflictor->s.origin, radius)) != NULL)
	{
		if (inflictor==ent)
			continue;
		if (!ent->takedamage)
			continue;

		if (ent->client)
			if (ent->client->pers.inventory[ITEM_INDEX(FindItem("Body Armor"))])
				continue;

		VectorAdd (ent->mins, ent->maxs, v);
		VectorMA (ent->s.origin, 0.5, v, v);
		VectorSubtract (inflictor->s.origin, v, v);
		if (CanDamage (ent, inflictor) && !ent->linked_flame)
		{
			ent->linked_flame=inflictor;
			inflictor->flamed=ent;
			inflictor->classname = "linkedflame";
			return;
		}
	}

}

qboolean Check_RadiusFire (vec3_t start)
{
	float	radius = 100;
	edict_t	*ent = NULL;

	while ((ent = findradius(ent, start, radius)) != NULL)
	{
		if (!Q_stricmp (ent->classname, "tossedflame"))
			return false;
	}
	return true;
}